/*********************************************************************
*	 项目名称:MG82F6P32-DEMO
*			测试板使用 TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV Board
*			CpuCLK=12MHz, SysCLK=12MHz
*			功能描述：
*				UART1中断发送及接收
*			注意事项：
*
*	 建立时间: 2024.2.26
*	 修改日志:
*
*********************************************************************/
#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*定义系统时钟SysClk (MAX.50MHz)
*可选:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*定义CPU时钟 CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	/256)
#define	TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	%256)


#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26

#define	UART1_RX_BUFF_SIZE	 32			 //字节
#define	UART1_TX_BUFF_SIZE	 32			 //字节
xdata u8 RcvBuf[UART1_RX_BUFF_SIZE];
u8 Uart1RxIn =0;
u8 Uart1RxOut =0;
xdata u8 TxBuf[UART1_TX_BUFF_SIZE];
u8 Uart1TxIn =0;
u8 Uart1TxOut =0;
bit	bUart1TxFlag;

u8 LedTime;

/***********************************************************************************
*函数名称:	 void INT_UART1(void)
*功能描述:UART1	中断服务程序

*输入参数:
*返回参数:
*************************************************************************************/
void INT_UART1(void) interrupt INT_VECTOR_UART1
{
	_push_(SFRPI);		   //保护SFRPI寄存器值

	SFR_SetPage(1);		   //S1CON仅1页操作
	if(TI1)
	{
	   TI1 = 0;
		if(Uart1TxIn==Uart1TxOut)
		{
			bUart1TxFlag=FALSE;
		}
		else
		{
			S1BUF=TxBuf[Uart1TxOut];
			bUart1TxFlag=TRUE;
			Uart1TxOut++;
			if(Uart1TxOut>=UART1_TX_BUFF_SIZE)
			{
				Uart1TxOut=0;
			}
		}
	}
	if(RI1)
	{
		RI1	= 0;
		RcvBuf[Uart1RxIn] =	S1BUF;
		Uart1RxIn++;
		if(Uart1RxIn >=UART1_RX_BUFF_SIZE)
		{
			Uart1RxIn =0;
		}

	}
	_pop_(SFRPI);		   //恢复SFRPI寄存器值
}

/***********************************************************************************
*函数名称:	 void INT_T0(void)
*功能描述:T0 中断服务程序
*输入参数:
*返回参数:
*************************************************************************************/
void INT_T0(void) interrupt	INT_VECTOR_T0
{
	TH0=TIMER_12T_1ms_TH;
	TL0=TIMER_12T_1ms_TL;

	if(LedTime!=0) LedTime--;
}

/***********************************************************************************
*函数名称:void Uart1SendByte(u8	tByte)
*功能描述:Uart1发送数据
*输入参数:u8 tByte:	待发送的数据
*返回参数:
*************************************************************************************/
void Uart1SendByte(u8 tByte)
{
	u8 i;

	if(bUart1TxFlag==FALSE)
	{
		Uart1TxOut=0;
		Uart1TxIn=1;
		TxBuf[0]=tByte;
		SFR_SetPage(1);
		TI1=1;
		SFR_SetPage(0);
	}
	else
	{
		i=Uart1TxIn;
		TxBuf[i]=tByte;
		i++;
		if(i>=UART1_TX_BUFF_SIZE)
		{
			i=0;
		}
		while(i==Uart1TxOut)
		{
		}
		INT_DisUART1();
		Uart1TxIn=i;
		INT_EnUART1();
	}
}

/***********************************************************************************
*函数名称:void Uart1SendStr(u8*	PStr)
*功能描述:Uart1发送字符串
*输入参数: u8* PStr:字符串起始地址
*返回参数:
*************************************************************************************/
void Uart1SendStr(u8* PStr)
{
	while(*PStr	!= 0)
	{
		Uart1SendByte(*PStr);
		PStr ++;
	}
}

/*************************************************
*函数名称:	   void	DelayXus(u16 xUs)
*功能描述:		延时程序，单位为us
*输入参数:	   u8 Us ->	*1us  (1~255)
*返回参数:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*函数名称:	   void	DelayXms(u16 xMs)
*功能描述:	   延时程序，单位为ms
*输入参数:	   u16 xMs -> *1ms	(1~65535)
*返回参数:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*函数名称:	 void InitUart1(void)
*功能描述:	 Uart1初始化(波特率9600)、RX:P34 TX:P35
*输入参数:
*返回参数:
*************************************************************************************/
void InitUart1(void)
{
	UART1_SetMode8bitUARTVar();								// 8位 可变波特率

	UART1_EnS1BRG();										//使能S1BRG波特率发生器
	UART1_SetBaudRateX2();									//使能双倍波特率
	UART1_SetRXD1TXD1_P34P35();								//设置串口使用引脚：RX:P34 TX:P35
	UART1_EnReception();									//使能串口接收功能
	UART1_SetS1BRGSelSYSCLK();								//设置S1BRG时钟源为系统时钟：1T

	UART1_SetS1BRGValue(S1BRG_BRGRL_9600_2X_12000000_1T);	//设置S1BRG值及重载值

}

/***********************************************************************************
*函数名称:	 void InitPort(void)
*功能描述:	 IO配置
*输入参数:
*返回参数:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);				// 设置P22,P24,P26为推挽输出(LED)
	PORT_SetP3QuasiBi(BIT4|BIT5);					// 设置P34,P35为准双向口
}

/***********************************************************************************
*函数名称:	 void InitClock(void)
*功能描述:	 时钟初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 输出 MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*函数名称:	 void InitInterrupt(void)
*功能描述:	 中断初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnTIMER0();			//	使能T0中断
	INT_EnUART1();			//	使能串口中断
	INT_EnAll();			//	使能全局中断

}

/***********************************************************************************
*函数名称:	 void InitTimer0(void)
*功能描述:Timer0初始化设置
*输入参数:
*返回参数:
*************************************************************************************/
void InitTimer0(void)
{
	TM_SetT0Mode_1_16BIT_TIMER();			// 设置T0模式为16位模式
	TM_SetT0Clock_SYSCLKDiv12();			// 设置T0时钟源为	SYSCLK/12
	TM_SetT0Gate_Disable();

	TM_SetT0LowByte(TIMER_12T_1ms_TL);		// 设置T0低8位
	TM_SetT0HighByte(TIMER_12T_1ms_TH);		// 设置T0高8位

	TM_EnableT0();							// 使能T0
}

/***********************************************************************************
*函数名称:	 void InitSystem(void)
*功能描述:	 系统初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitTimer0();
	InitUart1();
	InitInterrupt();
}

void main()
{
	InitSystem();


	Uart1RxIn=0;
	Uart1RxOut=0;
	Uart1TxIn=0;
	Uart1TxOut=0;
	bUart1TxFlag=0;

	IO_LED_G=0;
	DelayXms(1000);
	IO_LED_G=1;
	DelayXms(500);

	Uart1SendStr("Start!\n");


	while(1)
	{
		if(LedTime==0)
		{
			LedTime=100;
			 IO_LED_G =	!IO_LED_G;
		}
		if(Uart1RxIn !=	Uart1RxOut)
		{
			Uart1SendByte(RcvBuf[Uart1RxOut]);
			Uart1RxOut++;
			if(Uart1RxOut>=UART1_RX_BUFF_SIZE)
			{
				Uart1RxOut=0;
			}
		}

	}

}


